"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _os = _interopRequireDefault(require("os"));

var _dns = _interopRequireDefault(require("dns"));

var _constants = _interopRequireDefault(require("constants"));

var _stream = require("stream");

var _identity = require("@azure/identity");

var _bulkLoad = _interopRequireDefault(require("./bulk-load"));

var _debug = _interopRequireDefault(require("./debug"));

var _events = require("events");

var _instanceLookup = require("./instance-lookup");

var _transientErrorLookup = require("./transient-error-lookup");

var _packet = require("./packet");

var _preloginPayload = _interopRequireDefault(require("./prelogin-payload"));

var _login7Payload = _interopRequireDefault(require("./login7-payload"));

var _ntlmPayload = _interopRequireDefault(require("./ntlm-payload"));

var _request = _interopRequireDefault(require("./request"));

var _rpcrequestPayload = _interopRequireDefault(require("./rpcrequest-payload"));

var _sqlbatchPayload = _interopRequireDefault(require("./sqlbatch-payload"));

var _messageIo = _interopRequireDefault(require("./message-io"));

var _tokenStreamParser = require("./token/token-stream-parser");

var _transaction = require("./transaction");

var _errors = require("./errors");

var _connector = require("./connector");

var _library = require("./library");

var _tdsVersions = require("./tds-versions");

var _message = _interopRequireDefault(require("./message"));

var _ntlm = require("./ntlm");

var _nodeAbortController = require("node-abort-controller");

var _dataType = require("./data-type");

var _bulkLoadPayload = require("./bulk-load-payload");

var _esAggregateError = _interopRequireDefault(require("es-aggregate-error"));

var _package = require("../package.json");

var _url = require("url");

var _handler = require("./token/handler");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @private
 */
const KEEP_ALIVE_INITIAL_DELAY = 30 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CLIENT_REQUEST_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CANCEL_TIMEOUT = 5 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_RETRY_INTERVAL = 500;
/**
 * @private
 */

const DEFAULT_PACKET_SIZE = 4 * 1024;
/**
 * @private
 */

const DEFAULT_TEXTSIZE = 2147483647;
/**
 * @private
 */

const DEFAULT_DATEFIRST = 7;
/**
 * @private
 */

const DEFAULT_PORT = 1433;
/**
 * @private
 */

const DEFAULT_TDS_VERSION = '7_4';
/**
 * @private
 */

const DEFAULT_LANGUAGE = 'us_english';
/**
 * @private
 */

const DEFAULT_DATEFORMAT = 'mdy';

/**
 * @private
 */
const CLEANUP_TYPE = {
  NORMAL: 0,
  REDIRECT: 1,
  RETRY: 2
};

/**
 * A [[Connection]] instance represents a single connection to a database server.
 *
 * ```js
 * var Connection = require('tedious').Connection;
 * var config = {
 *  "authentication": {
 *    ...,
 *    "options": {...}
 *  },
 *  "options": {...}
 * };
 * var connection = new Connection(config);
 * ```
 *
 * Only one request at a time may be executed on a connection. Once a [[Request]]
 * has been initiated (with [[Connection.callProcedure]], [[Connection.execSql]],
 * or [[Connection.execSqlBatch]]), another should not be initiated until the
 * [[Request]]'s completion callback is called.
 */
class Connection extends _events.EventEmitter {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * Note: be aware of the different options field:
   * 1. config.authentication.options
   * 2. config.options
   *
   * ```js
   * const { Connection } = require('tedious');
   *
   * const config = {
   *  "authentication": {
   *    ...,
   *    "options": {...}
   *  },
   *  "options": {...}
   * };
   *
   * const connection = new Connection(config);
   * ```
   *
   * @param config
   */
  constructor(config) {
    super();
    this.fedAuthRequired = void 0;
    this.config = void 0;
    this.secureContextOptions = void 0;
    this.inTransaction = void 0;
    this.transactionDescriptors = void 0;
    this.transactionDepth = void 0;
    this.isSqlBatch = void 0;
    this.curTransientRetryCount = void 0;
    this.transientErrorLookup = void 0;
    this.closed = void 0;
    this.loginError = void 0;
    this.debug = void 0;
    this.ntlmpacket = void 0;
    this.ntlmpacketBuffer = void 0;
    this.routingData = void 0;
    this.messageIo = void 0;
    this.state = void 0;
    this.resetConnectionOnNextRequest = void 0;
    this.request = void 0;
    this.procReturnStatusValue = void 0;
    this.socket = void 0;
    this.messageBuffer = void 0;
    this.connectTimer = void 0;
    this.cancelTimer = void 0;
    this.requestTimer = void 0;
    this.retryTimer = void 0;
    this._cancelAfterRequestSent = void 0;
    this.databaseCollation = void 0;

    if (typeof config !== 'object' || config === null) {
      throw new TypeError('The "config" argument is required and must be of type Object.');
    }

    if (typeof config.server !== 'string') {
      throw new TypeError('The "config.server" property is required and must be of type string.');
    }

    this.fedAuthRequired = false;
    let authentication;

    if (config.authentication !== undefined) {
      if (typeof config.authentication !== 'object' || config.authentication === null) {
        throw new TypeError('The "config.authentication" property must be of type Object.');
      }

      const type = config.authentication.type;
      const options = config.authentication.options === undefined ? {} : config.authentication.options;

      if (typeof type !== 'string') {
        throw new TypeError('The "config.authentication.type" property must be of type string.');
      }

      if (type !== 'default' && type !== 'ntlm' && type !== 'azure-active-directory-password' && type !== 'azure-active-directory-access-token' && type !== 'azure-active-directory-msi-vm' && type !== 'azure-active-directory-msi-app-service' && type !== 'azure-active-directory-service-principal-secret' && type !== 'azure-active-directory-default') {
        throw new TypeError('The "type" property must one of "default", "ntlm", "azure-active-directory-password", "azure-active-directory-access-token", "azure-active-directory-default", "azure-active-directory-msi-vm" or "azure-active-directory-msi-app-service" or "azure-active-directory-service-principal-secret".');
      }

      if (typeof options !== 'object' || options === null) {
        throw new TypeError('The "config.authentication.options" property must be of type object.');
      }

      if (type === 'ntlm') {
        if (typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        }

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'ntlm',
          options: {
            userName: options.userName,
            password: options.password,
            domain: options.domain && options.domain.toUpperCase()
          }
        };
      } else if (type === 'azure-active-directory-password') {
        if (typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        if (options.tenantId !== undefined && typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-password',
          options: {
            userName: options.userName,
            password: options.password,
            tenantId: options.tenantId,
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-access-token') {
        if (typeof options.token !== 'string') {
          throw new TypeError('The "config.authentication.options.token" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-access-token',
          options: {
            token: options.token
          }
        };
      } else if (type === 'azure-active-directory-msi-vm') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-vm',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-default') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-default',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-msi-app-service') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-app-service',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-service-principal-secret') {
        if (typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        if (typeof options.clientSecret !== 'string') {
          throw new TypeError('The "config.authentication.options.clientSecret" property must be of type string.');
        }

        if (typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-service-principal-secret',
          options: {
            clientId: options.clientId,
            clientSecret: options.clientSecret,
            tenantId: options.tenantId
          }
        };
      } else {
        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'default',
          options: {
            userName: options.userName,
            password: options.password
          }
        };
      }
    } else {
      authentication = {
        type: 'default',
        options: {
          userName: undefined,
          password: undefined
        }
      };
    }

    this.config = {
      server: config.server,
      authentication: authentication,
      options: {
        abortTransactionOnError: false,
        appName: undefined,
        camelCaseColumns: false,
        cancelTimeout: DEFAULT_CANCEL_TIMEOUT,
        columnEncryptionKeyCacheTTL: 2 * 60 * 60 * 1000,
        // Units: miliseconds
        columnEncryptionSetting: false,
        columnNameReplacer: undefined,
        connectionRetryInterval: DEFAULT_CONNECT_RETRY_INTERVAL,
        connectTimeout: DEFAULT_CONNECT_TIMEOUT,
        connectionIsolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        cryptoCredentialsDetails: {},
        database: undefined,
        datefirst: DEFAULT_DATEFIRST,
        dateFormat: DEFAULT_DATEFORMAT,
        debug: {
          data: false,
          packet: false,
          payload: false,
          token: false
        },
        enableAnsiNull: true,
        enableAnsiNullDefault: true,
        enableAnsiPadding: true,
        enableAnsiWarnings: true,
        enableArithAbort: true,
        enableConcatNullYieldsNull: true,
        enableCursorCloseOnCommit: null,
        enableImplicitTransactions: false,
        enableNumericRoundabort: false,
        enableQuotedIdentifier: true,
        encrypt: true,
        fallbackToDefaultDb: false,
        encryptionKeyStoreProviders: undefined,
        instanceName: undefined,
        isolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        language: DEFAULT_LANGUAGE,
        localAddress: undefined,
        maxRetriesOnTransientErrors: 3,
        multiSubnetFailover: false,
        packetSize: DEFAULT_PACKET_SIZE,
        port: DEFAULT_PORT,
        readOnlyIntent: false,
        requestTimeout: DEFAULT_CLIENT_REQUEST_TIMEOUT,
        rowCollectionOnDone: false,
        rowCollectionOnRequestCompletion: false,
        serverName: undefined,
        serverSupportsColumnEncryption: false,
        tdsVersion: DEFAULT_TDS_VERSION,
        textsize: DEFAULT_TEXTSIZE,
        trustedServerNameAE: undefined,
        trustServerCertificate: false,
        useColumnNames: false,
        useUTC: true,
        workstationId: undefined,
        lowerCaseGuids: false
      }
    };

    if (config.options) {
      if (config.options.port && config.options.instanceName) {
        throw new Error('Port and instanceName are mutually exclusive, but ' + config.options.port + ' and ' + config.options.instanceName + ' provided');
      }

      if (config.options.abortTransactionOnError !== undefined) {
        if (typeof config.options.abortTransactionOnError !== 'boolean' && config.options.abortTransactionOnError !== null) {
          throw new TypeError('The "config.options.abortTransactionOnError" property must be of type string or null.');
        }

        this.config.options.abortTransactionOnError = config.options.abortTransactionOnError;
      }

      if (config.options.appName !== undefined) {
        if (typeof config.options.appName !== 'string') {
          throw new TypeError('The "config.options.appName" property must be of type string.');
        }

        this.config.options.appName = config.options.appName;
      }

      if (config.options.camelCaseColumns !== undefined) {
        if (typeof config.options.camelCaseColumns !== 'boolean') {
          throw new TypeError('The "config.options.camelCaseColumns" property must be of type boolean.');
        }

        this.config.options.camelCaseColumns = config.options.camelCaseColumns;
      }

      if (config.options.cancelTimeout !== undefined) {
        if (typeof config.options.cancelTimeout !== 'number') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type number.');
        }

        this.config.options.cancelTimeout = config.options.cancelTimeout;
      }

      if (config.options.columnNameReplacer) {
        if (typeof config.options.columnNameReplacer !== 'function') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type function.');
        }

        this.config.options.columnNameReplacer = config.options.columnNameReplacer;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.connectionIsolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.connectionIsolationLevel, 'config.options.connectionIsolationLevel');
        this.config.options.connectionIsolationLevel = config.options.connectionIsolationLevel;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.cryptoCredentialsDetails !== undefined) {
        if (typeof config.options.cryptoCredentialsDetails !== 'object' || config.options.cryptoCredentialsDetails === null) {
          throw new TypeError('The "config.options.cryptoCredentialsDetails" property must be of type Object.');
        }

        this.config.options.cryptoCredentialsDetails = config.options.cryptoCredentialsDetails;
      }

      if (config.options.database !== undefined) {
        if (typeof config.options.database !== 'string') {
          throw new TypeError('The "config.options.database" property must be of type string.');
        }

        this.config.options.database = config.options.database;
      }

      if (config.options.datefirst !== undefined) {
        if (typeof config.options.datefirst !== 'number' && config.options.datefirst !== null) {
          throw new TypeError('The "config.options.datefirst" property must be of type number.');
        }

        if (config.options.datefirst !== null && (config.options.datefirst < 1 || config.options.datefirst > 7)) {
          throw new RangeError('The "config.options.datefirst" property must be >= 1 and <= 7');
        }

        this.config.options.datefirst = config.options.datefirst;
      }

      if (config.options.dateFormat !== undefined) {
        if (typeof config.options.dateFormat !== 'string' && config.options.dateFormat !== null) {
          throw new TypeError('The "config.options.dateFormat" property must be of type string or null.');
        }

        this.config.options.dateFormat = config.options.dateFormat;
      }

      if (config.options.debug) {
        if (config.options.debug.data !== undefined) {
          if (typeof config.options.debug.data !== 'boolean') {
            throw new TypeError('The "config.options.debug.data" property must be of type boolean.');
          }

          this.config.options.debug.data = config.options.debug.data;
        }

        if (config.options.debug.packet !== undefined) {
          if (typeof config.options.debug.packet !== 'boolean') {
            throw new TypeError('The "config.options.debug.packet" property must be of type boolean.');
          }

          this.config.options.debug.packet = config.options.debug.packet;
        }

        if (config.options.debug.payload !== undefined) {
          if (typeof config.options.debug.payload !== 'boolean') {
            throw new TypeError('The "config.options.debug.payload" property must be of type boolean.');
          }

          this.config.options.debug.payload = config.options.debug.payload;
        }

        if (config.options.debug.token !== undefined) {
          if (typeof config.options.debug.token !== 'boolean') {
            throw new TypeError('The "config.options.debug.token" property must be of type boolean.');
          }

          this.config.options.debug.token = config.options.debug.token;
        }
      }

      if (config.options.enableAnsiNull !== undefined) {
        if (typeof config.options.enableAnsiNull !== 'boolean' && config.options.enableAnsiNull !== null) {
          throw new TypeError('The "config.options.enableAnsiNull" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNull = config.options.enableAnsiNull;
      }

      if (config.options.enableAnsiNullDefault !== undefined) {
        if (typeof config.options.enableAnsiNullDefault !== 'boolean' && config.options.enableAnsiNullDefault !== null) {
          throw new TypeError('The "config.options.enableAnsiNullDefault" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNullDefault = config.options.enableAnsiNullDefault;
      }

      if (config.options.enableAnsiPadding !== undefined) {
        if (typeof config.options.enableAnsiPadding !== 'boolean' && config.options.enableAnsiPadding !== null) {
          throw new TypeError('The "config.options.enableAnsiPadding" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiPadding = config.options.enableAnsiPadding;
      }

      if (config.options.enableAnsiWarnings !== undefined) {
        if (typeof config.options.enableAnsiWarnings !== 'boolean' && config.options.enableAnsiWarnings !== null) {
          throw new TypeError('The "config.options.enableAnsiWarnings" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiWarnings = config.options.enableAnsiWarnings;
      }

      if (config.options.enableArithAbort !== undefined) {
        if (typeof config.options.enableArithAbort !== 'boolean' && config.options.enableArithAbort !== null) {
          throw new TypeError('The "config.options.enableArithAbort" property must be of type boolean or null.');
        }

        this.config.options.enableArithAbort = config.options.enableArithAbort;
      }

      if (config.options.enableConcatNullYieldsNull !== undefined) {
        if (typeof config.options.enableConcatNullYieldsNull !== 'boolean' && config.options.enableConcatNullYieldsNull !== null) {
          throw new TypeError('The "config.options.enableConcatNullYieldsNull" property must be of type boolean or null.');
        }

        this.config.options.enableConcatNullYieldsNull = config.options.enableConcatNullYieldsNull;
      }

      if (config.options.enableCursorCloseOnCommit !== undefined) {
        if (typeof config.options.enableCursorCloseOnCommit !== 'boolean' && config.options.enableCursorCloseOnCommit !== null) {
          throw new TypeError('The "config.options.enableCursorCloseOnCommit" property must be of type boolean or null.');
        }

        this.config.options.enableCursorCloseOnCommit = config.options.enableCursorCloseOnCommit;
      }

      if (config.options.enableImplicitTransactions !== undefined) {
        if (typeof config.options.enableImplicitTransactions !== 'boolean' && config.options.enableImplicitTransactions !== null) {
          throw new TypeError('The "config.options.enableImplicitTransactions" property must be of type boolean or null.');
        }

        this.config.options.enableImplicitTransactions = config.options.enableImplicitTransactions;
      }

      if (config.options.enableNumericRoundabort !== undefined) {
        if (typeof config.options.enableNumericRoundabort !== 'boolean' && config.options.enableNumericRoundabort !== null) {
          throw new TypeError('The "config.options.enableNumericRoundabort" property must be of type boolean or null.');
        }

        this.config.options.enableNumericRoundabort = config.options.enableNumericRoundabort;
      }

      if (config.options.enableQuotedIdentifier !== undefined) {
        if (typeof config.options.enableQuotedIdentifier !== 'boolean' && config.options.enableQuotedIdentifier !== null) {
          throw new TypeError('The "config.options.enableQuotedIdentifier" property must be of type boolean or null.');
        }

        this.config.options.enableQuotedIdentifier = config.options.enableQuotedIdentifier;
      }

      if (config.options.encrypt !== undefined) {
        if (typeof config.options.encrypt !== 'boolean') {
          throw new TypeError('The "config.options.encrypt" property must be of type boolean.');
        }

        this.config.options.encrypt = config.options.encrypt;
      }

      if (config.options.fallbackToDefaultDb !== undefined) {
        if (typeof config.options.fallbackToDefaultDb !== 'boolean') {
          throw new TypeError('The "config.options.fallbackToDefaultDb" property must be of type boolean.');
        }

        this.config.options.fallbackToDefaultDb = config.options.fallbackToDefaultDb;
      }

      if (config.options.instanceName !== undefined) {
        if (typeof config.options.instanceName !== 'string') {
          throw new TypeError('The "config.options.instanceName" property must be of type string.');
        }

        this.config.options.instanceName = config.options.instanceName;
        this.config.options.port = undefined;
      }

      if (config.options.isolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.isolationLevel, 'config.options.isolationLevel');
        this.config.options.isolationLevel = config.options.isolationLevel;
      }

      if (config.options.language !== undefined) {
        if (typeof config.options.language !== 'string' && config.options.language !== null) {
          throw new TypeError('The "config.options.language" property must be of type string or null.');
        }

        this.config.options.language = config.options.language;
      }

      if (config.options.localAddress !== undefined) {
        if (typeof config.options.localAddress !== 'string') {
          throw new TypeError('The "config.options.localAddress" property must be of type string.');
        }

        this.config.options.localAddress = config.options.localAddress;
      }

      if (config.options.multiSubnetFailover !== undefined) {
        if (typeof config.options.multiSubnetFailover !== 'boolean') {
          throw new TypeError('The "config.options.multiSubnetFailover" property must be of type boolean.');
        }

        this.config.options.multiSubnetFailover = config.options.multiSubnetFailover;
      }

      if (config.options.packetSize !== undefined) {
        if (typeof config.options.packetSize !== 'number') {
          throw new TypeError('The "config.options.packetSize" property must be of type number.');
        }

        this.config.options.packetSize = config.options.packetSize;
      }

      if (config.options.port !== undefined) {
        if (typeof config.options.port !== 'number') {
          throw new TypeError('The "config.options.port" property must be of type number.');
        }

        if (config.options.port <= 0 || config.options.port >= 65536) {
          throw new RangeError('The "config.options.port" property must be > 0 and < 65536');
        }

        this.config.options.port = config.options.port;
        this.config.options.instanceName = undefined;
      }

      if (config.options.readOnlyIntent !== undefined) {
        if (typeof config.options.readOnlyIntent !== 'boolean') {
          throw new TypeError('The "config.options.readOnlyIntent" property must be of type boolean.');
        }

        this.config.options.readOnlyIntent = config.options.readOnlyIntent;
      }

      if (config.options.requestTimeout !== undefined) {
        if (typeof config.options.requestTimeout !== 'number') {
          throw new TypeError('The "config.options.requestTimeout" property must be of type number.');
        }

        this.config.options.requestTimeout = config.options.requestTimeout;
      }

      if (config.options.maxRetriesOnTransientErrors !== undefined) {
        if (typeof config.options.maxRetriesOnTransientErrors !== 'number') {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be of type number.');
        }

        if (config.options.maxRetriesOnTransientErrors < 0) {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be equal or greater than 0.');
        }

        this.config.options.maxRetriesOnTransientErrors = config.options.maxRetriesOnTransientErrors;
      }

      if (config.options.connectionRetryInterval !== undefined) {
        if (typeof config.options.connectionRetryInterval !== 'number') {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be of type number.');
        }

        if (config.options.connectionRetryInterval <= 0) {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be greater than 0.');
        }

        this.config.options.connectionRetryInterval = config.options.connectionRetryInterval;
      }

      if (config.options.rowCollectionOnDone !== undefined) {
        if (typeof config.options.rowCollectionOnDone !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnDone" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnDone = config.options.rowCollectionOnDone;
      }

      if (config.options.rowCollectionOnRequestCompletion !== undefined) {
        if (typeof config.options.rowCollectionOnRequestCompletion !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnRequestCompletion" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnRequestCompletion = config.options.rowCollectionOnRequestCompletion;
      }

      if (config.options.tdsVersion !== undefined) {
        if (typeof config.options.tdsVersion !== 'string') {
          throw new TypeError('The "config.options.tdsVersion" property must be of type string.');
        }

        this.config.options.tdsVersion = config.options.tdsVersion;
      }

      if (config.options.textsize !== undefined) {
        if (typeof config.options.textsize !== 'number' && config.options.textsize !== null) {
          throw new TypeError('The "config.options.textsize" property must be of type number or null.');
        }

        if (config.options.textsize > 2147483647) {
          throw new TypeError('The "config.options.textsize" can\'t be greater than 2147483647.');
        } else if (config.options.textsize < -1) {
          throw new TypeError('The "config.options.textsize" can\'t be smaller than -1.');
        }

        this.config.options.textsize = config.options.textsize | 0;
      }

      if (config.options.trustServerCertificate !== undefined) {
        if (typeof config.options.trustServerCertificate !== 'boolean') {
          throw new TypeError('The "config.options.trustServerCertificate" property must be of type boolean.');
        }

        this.config.options.trustServerCertificate = config.options.trustServerCertificate;
      }

      if (config.options.useColumnNames !== undefined) {
        if (typeof config.options.useColumnNames !== 'boolean') {
          throw new TypeError('The "config.options.useColumnNames" property must be of type boolean.');
        }

        this.config.options.useColumnNames = config.options.useColumnNames;
      }

      if (config.options.useUTC !== undefined) {
        if (typeof config.options.useUTC !== 'boolean') {
          throw new TypeError('The "config.options.useUTC" property must be of type boolean.');
        }

        this.config.options.useUTC = config.options.useUTC;
      }

      if (config.options.workstationId !== undefined) {
        if (typeof config.options.workstationId !== 'string') {
          throw new TypeError('The "config.options.workstationId" property must be of type string.');
        }

        this.config.options.workstationId = config.options.workstationId;
      }

      if (config.options.lowerCaseGuids !== undefined) {
        if (typeof config.options.lowerCaseGuids !== 'boolean') {
          throw new TypeError('The "config.options.lowerCaseGuids" property must be of type boolean.');
        }

        this.config.options.lowerCaseGuids = config.options.lowerCaseGuids;
      }
    }

    this.secureContextOptions = this.config.options.cryptoCredentialsDetails;

    if (this.secureContextOptions.secureOptions === undefined) {
      // If the caller has not specified their own `secureOptions`,
      // we set `SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS` here.
      // Older SQL Server instances running on older Windows versions have
      // trouble with the BEAST workaround in OpenSSL.
      // As BEAST is a browser specific exploit, we can just disable this option here.
      this.secureContextOptions = Object.create(this.secureContextOptions, {
        secureOptions: {
          value: _constants.default.SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS
        }
      });
    }

    this.debug = this.createDebug();
    this.inTransaction = false;
    this.transactionDescriptors = [Buffer.from([0, 0, 0, 0, 0, 0, 0, 0])]; // 'beginTransaction', 'commitTransaction' and 'rollbackTransaction'
    // events are utilized to maintain inTransaction property state which in
    // turn is used in managing transactions. These events are only fired for
    // TDS version 7.2 and beyond. The properties below are used to emulate
    // equivalent behavior for TDS versions before 7.2.

    this.transactionDepth = 0;
    this.isSqlBatch = false;
    this.closed = false;
    this.messageBuffer = Buffer.alloc(0);
    this.curTransientRetryCount = 0;
    this.transientErrorLookup = new _transientErrorLookup.TransientErrorLookup();
    this.state = this.STATE.INITIALIZED;

    this._cancelAfterRequestSent = () => {
      this.messageIo.sendMessage(_packet.TYPE.ATTENTION);
      this.createCancelTimer();
    };
  }

  connect(connectListener) {
    if (this.state !== this.STATE.INITIALIZED) {
      throw new _errors.ConnectionError('`.connect` can not be called on a Connection in `' + this.state.name + '` state.');
    }

    if (connectListener) {
      const onConnect = err => {
        this.removeListener('error', onError);
        connectListener(err);
      };

      const onError = err => {
        this.removeListener('connect', onConnect);
        connectListener(err);
      };

      this.once('connect', onConnect);
      this.once('error', onError);
    }

    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * The server has reported that the charset has changed.
   */


  on(event, listener) {
    return super.on(event, listener);
  }
  /**
   * @private
   */


  emit(event, ...args) {
    return super.emit(event, ...args);
  }
  /**
   * Closes the connection to the database.
   *
   * The [[Event_end]] will be emitted once the connection has been closed.
   */


  close() {
    this.transitionTo(this.STATE.FINAL);
  }
  /**
   * @private
   */


  initialiseConnection() {
    const signal = this.createConnectTimer();

    if (this.config.options.port) {
      return this.connectOnPort(this.config.options.port, this.config.options.multiSubnetFailover, signal);
    } else {
      return (0, _instanceLookup.instanceLookup)({
        server: this.config.server,
        instanceName: this.config.options.instanceName,
        timeout: this.config.options.connectTimeout,
        signal: signal
      }).then(port => {
        process.nextTick(() => {
          this.connectOnPort(port, this.config.options.multiSubnetFailover, signal);
        });
      }, err => {
        if (err.name === 'AbortError') {
          // Ignore the AbortError for now, this is still handled by the connectTimer firing
          return;
        }

        process.nextTick(() => {
          this.emit('connect', new _errors.ConnectionError(err.message, 'EINSTLOOKUP'));
        });
      });
    }
  }
  /**
   * @private
   */


  cleanupConnection(cleanupType) {
    if (!this.closed) {
      this.clearConnectTimer();
      this.clearRequestTimer();
      this.clearRetryTimer();
      this.closeConnection();

      if (cleanupType === CLEANUP_TYPE.REDIRECT) {
        this.emit('rerouting');
      } else if (cleanupType !== CLEANUP_TYPE.RETRY) {
        process.nextTick(() => {
          this.emit('end');
        });
      }

      const request = this.request;

      if (request) {
        const err = new _errors.RequestError('Connection closed before request completed.', 'ECLOSE');
        request.callback(err);
        this.request = undefined;
      }

      this.closed = true;
      this.loginError = undefined;
    }
  }
  /**
   * @private
   */


  createDebug() {
    const debug = new _debug.default(this.config.options.debug);
    debug.on('debug', message => {
      this.emit('debug', message);
    });
    return debug;
  }
  /**
   * @private
   */


  createTokenStreamParser(message, handler) {
    return new _tokenStreamParser.Parser(message, this.debug, handler, this.config.options);
  }

  connectOnPort(port, multiSubnetFailover, signal) {
    const connectOpts = {
      host: this.routingData ? this.routingData.server : this.config.server,
      port: this.routingData ? this.routingData.port : port,
      localAddress: this.config.options.localAddress
    };
    const connect = multiSubnetFailover ? _connector.connectInParallel : _connector.connectInSequence;
    connect(connectOpts, _dns.default.lookup, signal).then(socket => {
      process.nextTick(() => {
        socket.on('error', error => {
          this.socketError(error);
        });
        socket.on('close', () => {
          this.socketClose();
        });
        socket.on('end', () => {
          this.socketEnd();
        });
        socket.setKeepAlive(true, KEEP_ALIVE_INITIAL_DELAY);
        this.messageIo = new _messageIo.default(socket, this.config.options.packetSize, this.debug);
        this.messageIo.on('secure', cleartext => {
          this.emit('secure', cleartext);
        });
        this.socket = socket;
        this.closed = false;
        this.debug.log('connected to ' + this.config.server + ':' + this.config.options.port);
        this.sendPreLogin();
        this.transitionTo(this.STATE.SENT_PRELOGIN);
      });
    }, err => {
      if (err.name === 'AbortError') {
        return;
      }

      process.nextTick(() => {
        this.socketError(err);
      });
    });
  }
  /**
   * @private
   */


  closeConnection() {
    if (this.socket) {
      this.socket.destroy();
    }
  }
  /**
   * @private
   */


  createConnectTimer() {
    const controller = new _nodeAbortController.AbortController();
    this.connectTimer = setTimeout(() => {
      controller.abort();
      this.connectTimeout();
    }, this.config.options.connectTimeout);
    return controller.signal;
  }
  /**
   * @private
   */


  createCancelTimer() {
    this.clearCancelTimer();
    const timeout = this.config.options.cancelTimeout;

    if (timeout > 0) {
      this.cancelTimer = setTimeout(() => {
        this.cancelTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRequestTimer() {
    this.clearRequestTimer(); // release old timer, just to be safe

    const request = this.request;
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;

    if (timeout) {
      this.requestTimer = setTimeout(() => {
        this.requestTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRetryTimer() {
    this.clearRetryTimer();
    this.retryTimer = setTimeout(() => {
      this.retryTimeout();
    }, this.config.options.connectionRetryInterval);
  }
  /**
   * @private
   */


  connectTimeout() {
    const message = `Failed to connect to ${this.config.server}${this.config.options.port ? `:${this.config.options.port}` : `\\${this.config.options.instanceName}`} in ${this.config.options.connectTimeout}ms`;
    this.debug.log(message);
    this.emit('connect', new _errors.ConnectionError(message, 'ETIMEOUT'));
    this.connectTimer = undefined;
    this.dispatchEvent('connectTimeout');
  }
  /**
   * @private
   */


  cancelTimeout() {
    const message = `Failed to cancel request in ${this.config.options.cancelTimeout}ms`;
    this.debug.log(message);
    this.dispatchEvent('socketError', new _errors.ConnectionError(message, 'ETIMEOUT'));
  }
  /**
   * @private
   */


  requestTimeout() {
    this.requestTimer = undefined;
    const request = this.request;
    request.cancel();
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;
    const message = 'Timeout: Request failed to complete in ' + timeout + 'ms';
    request.error = new _errors.RequestError(message, 'ETIMEOUT');
  }
  /**
   * @private
   */


  retryTimeout() {
    this.retryTimer = undefined;
    this.emit('retry');
    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * @private
   */


  clearConnectTimer() {
    if (this.connectTimer) {
      clearTimeout(this.connectTimer);
      this.connectTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearCancelTimer() {
    if (this.cancelTimer) {
      clearTimeout(this.cancelTimer);
      this.cancelTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRequestTimer() {
    if (this.requestTimer) {
      clearTimeout(this.requestTimer);
      this.requestTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRetryTimer() {
    if (this.retryTimer) {
      clearTimeout(this.retryTimer);
      this.retryTimer = undefined;
    }
  }
  /**
   * @private
   */


  transitionTo(newState) {
    if (this.state === newState) {
      this.debug.log('State is already ' + newState.name);
      return;
    }

    if (this.state && this.state.exit) {
      this.state.exit.call(this, newState);
    }

    this.debug.log('State change: ' + (this.state ? this.state.name : 'undefined') + ' -> ' + newState.name);
    this.state = newState;

    if (this.state.enter) {
      this.state.enter.apply(this);
    }
  }
  /**
   * @private
   */


  getEventHandler(eventName) {
    const handler = this.state.events[eventName];

    if (!handler) {
      throw new Error(`No event '${eventName}' in state '${this.state.name}'`);
    }

    return handler;
  }
  /**
   * @private
   */


  dispatchEvent(eventName, ...args) {
    const handler = this.state.events[eventName];

    if (handler) {
      handler.apply(this, args);
    } else {
      this.emit('error', new Error(`No event '${eventName}' in state '${this.state.name}'`));
      this.close();
    }
  }
  /**
   * @private
   */


  socketError(error) {
    if (this.state === this.STATE.CONNECTING || this.state === this.STATE.SENT_TLSSSLNEGOTIATION) {
      const message = `Failed to connect to ${this.config.server}:${this.config.options.port} - ${error.message}`;
      this.debug.log(message);
      this.emit('connect', new _errors.ConnectionError(message, 'ESOCKET'));
    } else {
      const message = `Connection lost - ${error.message}`;
      this.debug.log(message);
      this.emit('error', new _errors.ConnectionError(message, 'ESOCKET'));
    }

    this.dispatchEvent('socketError', error);
  }
  /**
   * @private
   */


  socketEnd() {
    this.debug.log('socket ended');

    if (this.state !== this.STATE.FINAL) {
      const error = new Error('socket hang up');
      error.code = 'ECONNRESET';
      this.socketError(error);
    }
  }
  /**
   * @private
   */


  socketClose() {
    this.debug.log('connection to ' + this.config.server + ':' + this.config.options.port + ' closed');

    if (this.state === this.STATE.REROUTING) {
      this.debug.log('Rerouting to ' + this.routingData.server + ':' + this.routingData.port);
      this.dispatchEvent('reconnect');
    } else if (this.state === this.STATE.TRANSIENT_FAILURE_RETRY) {
      const server = this.routingData ? this.routingData.server : this.config.server;
      const port = this.routingData ? this.routingData.port : this.config.options.port;
      this.debug.log('Retry after transient failure connecting to ' + server + ':' + port);
      this.dispatchEvent('retry');
    } else {
      this.transitionTo(this.STATE.FINAL);
    }
  }
  /**
   * @private
   */


  sendPreLogin() {
    const [, major, minor, build] = /^(\d+)\.(\d+)\.(\d+)/.exec(_package.version) ?? ['0.0.0', '0', '0', '0'];
    const payload = new _preloginPayload.default({
      encrypt: this.config.options.encrypt,
      version: {
        major: Number(major),
        minor: Number(minor),
        build: Number(build),
        subbuild: 0
      }
    });
    this.messageIo.sendMessage(_packet.TYPE.PRELOGIN, payload.data);
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  sendLogin7Packet() {
    const payload = new _login7Payload.default({
      tdsVersion: _tdsVersions.versions[this.config.options.tdsVersion],
      packetSize: this.config.options.packetSize,
      clientProgVer: 0,
      clientPid: process.pid,
      connectionId: 0,
      clientTimeZone: new Date().getTimezoneOffset(),
      clientLcid: 0x00000409
    });
    const {
      authentication
    } = this.config;

    switch (authentication.type) {
      case 'azure-active-directory-password':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'default'
        };
        break;

      case 'azure-active-directory-access-token':
        payload.fedAuth = {
          type: 'SECURITYTOKEN',
          echo: this.fedAuthRequired,
          fedAuthToken: authentication.options.token
        };
        break;

      case 'azure-active-directory-msi-vm':
      case 'azure-active-directory-default':
      case 'azure-active-directory-msi-app-service':
      case 'azure-active-directory-service-principal-secret':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'integrated'
        };
        break;

      case 'ntlm':
        payload.sspi = (0, _ntlm.createNTLMRequest)({
          domain: authentication.options.domain
        });
        break;

      default:
        payload.userName = authentication.options.userName;
        payload.password = authentication.options.password;
    }

    payload.hostname = this.config.options.workstationId || _os.default.hostname();
    payload.serverName = this.routingData ? this.routingData.server : this.config.server;
    payload.appName = this.config.options.appName || 'Tedious';
    payload.libraryName = _library.name;
    payload.language = this.config.options.language;
    payload.database = this.config.options.database;
    payload.clientId = Buffer.from([1, 2, 3, 4, 5, 6]);
    payload.readOnlyIntent = this.config.options.readOnlyIntent;
    payload.initDbFatal = !this.config.options.fallbackToDefaultDb;
    this.routingData = undefined;
    this.messageIo.sendMessage(_packet.TYPE.LOGIN7, payload.toBuffer());
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  sendFedAuthTokenMessage(token) {
    const accessTokenLen = Buffer.byteLength(token, 'ucs2');
    const data = Buffer.alloc(8 + accessTokenLen);
    let offset = 0;
    offset = data.writeUInt32LE(accessTokenLen + 4, offset);
    offset = data.writeUInt32LE(accessTokenLen, offset);
    data.write(token, offset, 'ucs2');
    this.messageIo.sendMessage(_packet.TYPE.FEDAUTH_TOKEN, data); // sent the fedAuth token message, the rest is similar to standard login 7

    this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
  }
  /**
   * @private
   */


  sendInitialSql() {
    const payload = new _sqlbatchPayload.default(this.getInitialSql(), this.currentTransactionDescriptor(), this.config.options);
    const message = new _message.default({
      type: _packet.TYPE.SQL_BATCH
    });
    this.messageIo.outgoingMessageStream.write(message);

    _stream.Readable.from(payload).pipe(message);
  }
  /**
   * @private
   */


  getInitialSql() {
    const options = [];

    if (this.config.options.enableAnsiNull === true) {
      options.push('set ansi_nulls on');
    } else if (this.config.options.enableAnsiNull === false) {
      options.push('set ansi_nulls off');
    }

    if (this.config.options.enableAnsiNullDefault === true) {
      options.push('set ansi_null_dflt_on on');
    } else if (this.config.options.enableAnsiNullDefault === false) {
      options.push('set ansi_null_dflt_on off');
    }

    if (this.config.options.enableAnsiPadding === true) {
      options.push('set ansi_padding on');
    } else if (this.config.options.enableAnsiPadding === false) {
      options.push('set ansi_padding off');
    }

    if (this.config.options.enableAnsiWarnings === true) {
      options.push('set ansi_warnings on');
    } else if (this.config.options.enableAnsiWarnings === false) {
      options.push('set ansi_warnings off');
    }

    if (this.config.options.enableArithAbort === true) {
      options.push('set arithabort on');
    } else if (this.config.options.enableArithAbort === false) {
      options.push('set arithabort off');
    }

    if (this.config.options.enableConcatNullYieldsNull === true) {
      options.push('set concat_null_yields_null on');
    } else if (this.config.options.enableConcatNullYieldsNull === false) {
      options.push('set concat_null_yields_null off');
    }

    if (this.config.options.enableCursorCloseOnCommit === true) {
      options.push('set cursor_close_on_commit on');
    } else if (this.config.options.enableCursorCloseOnCommit === false) {
      options.push('set cursor_close_on_commit off');
    }

    if (this.config.options.datefirst !== null) {
      options.push(`set datefirst ${this.config.options.datefirst}`);
    }

    if (this.config.options.dateFormat !== null) {
      options.push(`set dateformat ${this.config.options.dateFormat}`);
    }

    if (this.config.options.enableImplicitTransactions === true) {
      options.push('set implicit_transactions on');
    } else if (this.config.options.enableImplicitTransactions === false) {
      options.push('set implicit_transactions off');
    }

    if (this.config.options.language !== null) {
      options.push(`set language ${this.config.options.language}`);
    }

    if (this.config.options.enableNumericRoundabort === true) {
      options.push('set numeric_roundabort on');
    } else if (this.config.options.enableNumericRoundabort === false) {
      options.push('set numeric_roundabort off');
    }

    if (this.config.options.enableQuotedIdentifier === true) {
      options.push('set quoted_identifier on');
    } else if (this.config.options.enableQuotedIdentifier === false) {
      options.push('set quoted_identifier off');
    }

    if (this.config.options.textsize !== null) {
      options.push(`set textsize ${this.config.options.textsize}`);
    }

    if (this.config.options.connectionIsolationLevel !== null) {
      options.push(`set transaction isolation level ${this.getIsolationLevelText(this.config.options.connectionIsolationLevel)}`);
    }

    if (this.config.options.abortTransactionOnError === true) {
      options.push('set xact_abort on');
    } else if (this.config.options.abortTransactionOnError === false) {
      options.push('set xact_abort off');
    }

    return options.join('\n');
  }
  /**
   * @private
   */


  processedInitialSql() {
    this.clearConnectTimer();
    this.emit('connect');
  }
  /**
   * Execute the SQL batch represented by [[Request]].
   * There is no param support, and unlike [[Request.execSql]],
   * it is not likely that SQL Server will reuse the execution plan it generates for the SQL.
   *
   * In almost all cases, [[Request.execSql]] will be a better choice.
   *
   * @param request A [[Request]] object representing the request.
   */


  execSqlBatch(request) {
    this.makeRequest(request, _packet.TYPE.SQL_BATCH, new _sqlbatchPayload.default(request.sqlTextOrProcedure, this.currentTransactionDescriptor(), this.config.options));
  }
  /**
   *  Execute the SQL represented by [[Request]].
   *
   * As `sp_executesql` is used to execute the SQL, if the same SQL is executed multiples times
   * using this function, the SQL Server query optimizer is likely to reuse the execution plan it generates
   * for the first execution. This may also result in SQL server treating the request like a stored procedure
   * which can result in the [[Event_doneInProc]] or [[Event_doneProc]] events being emitted instead of the
   * [[Event_done]] event you might expect. Using [[execSqlBatch]] will prevent this from occurring but may have a negative performance impact.
   *
   * Beware of the way that scoping rules apply, and how they may [affect local temp tables](http://weblogs.sqlteam.com/mladenp/archive/2006/11/03/17197.aspx)
   * If you're running in to scoping issues, then [[execSqlBatch]] may be a better choice.
   * See also [issue #24](https://github.com/pekim/tedious/issues/24)
   *
   * @param request A [[Request]] object representing the request.
   */


  execSql(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'statement',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    if (request.parameters.length) {
      parameters.push({
        type: _dataType.TYPES.NVarChar,
        name: 'params',
        value: request.makeParamsParameter(request.parameters),
        output: false,
        length: undefined,
        precision: undefined,
        scale: undefined
      });
      parameters.push(...request.parameters);
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_executesql', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Creates a new BulkLoad instance.
   *
   * @param table The name of the table to bulk-insert into.
   * @param options A set of bulk load options.
   */


  newBulkLoad(table, callbackOrOptions, callback) {
    let options;

    if (callback === undefined) {
      callback = callbackOrOptions;
      options = {};
    } else {
      options = callbackOrOptions;
    }

    if (typeof options !== 'object') {
      throw new TypeError('"options" argument must be an object');
    }

    return new _bulkLoad.default(table, this.databaseCollation, this.config.options, options, callback);
  }
  /**
   * Execute a [[BulkLoad]].
   *
   * ```js
   * // We want to perform a bulk load into a table with the following format:
   * // CREATE TABLE employees (first_name nvarchar(255), last_name nvarchar(255), day_of_birth date);
   *
   * const bulkLoad = connection.newBulkLoad('employees', (err, rowCount) => {
   *   // ...
   * });
   *
   * // First, we need to specify the columns that we want to write to,
   * // and their definitions. These definitions must match the actual table,
   * // otherwise the bulk load will fail.
   * bulkLoad.addColumn('first_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('last_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('date_of_birth', TYPES.Date, { nullable: false });
   *
   * // Execute a bulk load with a predefined list of rows.
   * //
   * // Note that these rows are held in memory until the
   * // bulk load was performed, so if you need to write a large
   * // number of rows (e.g. by reading from a CSV file),
   * // passing an `AsyncIterable` is advisable to keep memory usage low.
   * connection.execBulkLoad(bulkLoad, [
   *   { 'first_name': 'Steve', 'last_name': 'Jobs', 'day_of_birth': new Date('02-24-1955') },
   *   { 'first_name': 'Bill', 'last_name': 'Gates', 'day_of_birth': new Date('10-28-1955') }
   * ]);
   * ```
   *
   * @param bulkLoad A previously created [[BulkLoad]].
   * @param rows A [[Iterable]] or [[AsyncIterable]] that contains the rows that should be bulk loaded.
   */


  execBulkLoad(bulkLoad, rows) {
    bulkLoad.executionStarted = true;

    if (rows) {
      if (bulkLoad.streamingMode) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that was put in streaming mode.");
      }

      if (bulkLoad.firstRowWritten) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that already has rows written to it.");
      }

      const rowStream = _stream.Readable.from(rows); // Destroy the packet transform if an error happens in the row stream,
      // e.g. if an error is thrown from within a generator or stream.


      rowStream.on('error', err => {
        bulkLoad.rowToPacketTransform.destroy(err);
      }); // Destroy the row stream if an error happens in the packet transform,
      // e.g. if the bulk load is cancelled.

      bulkLoad.rowToPacketTransform.on('error', err => {
        rowStream.destroy(err);
      });
      rowStream.pipe(bulkLoad.rowToPacketTransform);
    } else if (!bulkLoad.streamingMode) {
      // If the bulkload was not put into streaming mode by the user,
      // we end the rowToPacketTransform here for them.
      //
      // If it was put into streaming mode, it's the user's responsibility
      // to end the stream.
      bulkLoad.rowToPacketTransform.end();
    }

    const onCancel = () => {
      request.cancel();
    };

    const payload = new _bulkLoadPayload.BulkLoadPayload(bulkLoad);
    const request = new _request.default(bulkLoad.getBulkInsertSql(), error => {
      bulkLoad.removeListener('cancel', onCancel);

      if (error) {
        if (error.code === 'UNKNOWN') {
          error.message += ' This is likely because the schema of the BulkLoad does not match the schema of the table you are attempting to insert into.';
        }

        bulkLoad.error = error;
        bulkLoad.callback(error);
        return;
      }

      this.makeRequest(bulkLoad, _packet.TYPE.BULK_LOAD, payload);
    });
    bulkLoad.once('cancel', onCancel);
    this.execSqlBatch(request);
  }
  /**
   * Prepare the SQL represented by the request.
   *
   * The request can then be used in subsequent calls to
   * [[execute]] and [[unprepare]]
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type. Parameter values are ignored.
   */


  prepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      value: undefined,
      output: true,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'params',
      value: request.parameters.length ? request.makeParamsParameter(request.parameters) : null,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'stmt',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    request.preparing = true; // TODO: We need to clean up this event handler, otherwise this leaks memory

    request.on('returnValue', (name, value) => {
      if (name === 'handle') {
        request.handle = value;
      } else {
        request.error = new _errors.RequestError(`Tedious > Unexpected output parameter ${name} from sp_prepare`);
      }
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_prepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Release the SQL Server resources associated with a previously prepared request.
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type.
   *   Parameter values are ignored.
   */


  unprepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_unprepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Execute previously prepared SQL, using the supplied parameters.
   *
   * @param request A previously prepared [[Request]].
   * @param parameters  An object whose names correspond to the names of
   *   parameters that were added to the [[Request]] before it was prepared.
   *   The object's values are passed as the parameters' values when the
   *   request is executed.
   */


  execute(request, parameters) {
    const executeParameters = [];
    executeParameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    try {
      for (let i = 0, len = request.parameters.length; i < len; i++) {
        const parameter = request.parameters[i];
        executeParameters.push({ ...parameter,
          value: parameter.type.validate(parameters ? parameters[parameter.name] : null, this.databaseCollation)
        });
      }
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_execute', executeParameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Call a stored procedure represented by [[Request]].
   *
   * @param request A [[Request]] object representing the request.
   */


  callProcedure(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default(request.sqlTextOrProcedure, request.parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Start a transaction.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`
   *   is present.
   * @param isolationLevel The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  beginTransaction(callback, name = '', isolationLevel = this.config.options.isolationLevel) {
    (0, _transaction.assertValidIsolationLevel)(isolationLevel, 'isolationLevel');
    const transaction = new _transaction.Transaction(name, isolationLevel);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SET TRANSACTION ISOLATION LEVEL ' + transaction.isolationLevelToTSQL() + ';BEGIN TRAN ' + transaction.name, err => {
        this.transactionDepth++;

        if (this.transactionDepth === 1) {
          this.inTransaction = true;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, err => {
      return callback(err, this.currentTransactionDescriptor());
    });
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.beginPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Commit a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`is present.
   */


  commitTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('COMMIT TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.commitPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Rollback a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  rollbackTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('ROLLBACK TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.rollbackPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Set a savepoint within a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.\
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  saveTransaction(callback, name) {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SAVE TRAN ' + transaction.name, err => {
        this.transactionDepth++;
        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.savePayload(this.currentTransactionDescriptor()));
  }
  /**
   * Run the given callback after starting a transaction, and commit or
   * rollback the transaction afterwards.
   *
   * This is a helper that employs [[beginTransaction]], [[commitTransaction]],
   * [[rollbackTransaction]], and [[saveTransaction]] to greatly simplify the
   * use of database transactions and automatically handle transaction nesting.
   *
   * @param cb
   * @param isolationLevel
   *   The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  transaction(cb, isolationLevel) {
    if (typeof cb !== 'function') {
      throw new TypeError('`cb` must be a function');
    }

    const useSavepoint = this.inTransaction;

    const name = '_tedious_' + _crypto.default.randomBytes(10).toString('hex');

    const txDone = (err, done, ...args) => {
      if (err) {
        if (this.inTransaction && this.state === this.STATE.LOGGED_IN) {
          this.rollbackTransaction(txErr => {
            done(txErr || err, ...args);
          }, name);
        } else {
          done(err, ...args);
        }
      } else if (useSavepoint) {
        if (this.config.options.tdsVersion < '7_2') {
          this.transactionDepth--;
        }

        done(null, ...args);
      } else {
        this.commitTransaction(txErr => {
          done(txErr, ...args);
        }, name);
      }
    };

    if (useSavepoint) {
      return this.saveTransaction(err => {
        if (err) {
          return cb(err);
        }

        if (isolationLevel) {
          return this.execSqlBatch(new _request.default('SET transaction isolation level ' + this.getIsolationLevelText(isolationLevel), err => {
            return cb(err, txDone);
          }));
        } else {
          return cb(null, txDone);
        }
      }, name);
    } else {
      return this.beginTransaction(err => {
        if (err) {
          return cb(err);
        }

        return cb(null, txDone);
      }, name, isolationLevel);
    }
  }
  /**
   * @private
   */


  makeRequest(request, packetType, payload) {
    if (this.state !== this.STATE.LOGGED_IN) {
      const message = 'Requests can only be made in the ' + this.STATE.LOGGED_IN.name + ' state, not the ' + this.state.name + ' state';
      this.debug.log(message);
      request.callback(new _errors.RequestError(message, 'EINVALIDSTATE'));
    } else if (request.canceled) {
      process.nextTick(() => {
        request.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
      });
    } else {
      if (packetType === _packet.TYPE.SQL_BATCH) {
        this.isSqlBatch = true;
      } else {
        this.isSqlBatch = false;
      }

      this.request = request;
      request.connection = this;
      request.rowCount = 0;
      request.rows = [];
      request.rst = [];

      const onCancel = () => {
        payloadStream.unpipe(message);
        payloadStream.destroy(new _errors.RequestError('Canceled.', 'ECANCEL')); // set the ignore bit and end the message.

        message.ignore = true;
        message.end();

        if (request instanceof _request.default && request.paused) {
          // resume the request if it was paused so we can read the remaining tokens
          request.resume();
        }
      };

      request.once('cancel', onCancel);
      this.createRequestTimer();
      const message = new _message.default({
        type: packetType,
        resetConnection: this.resetConnectionOnNextRequest
      });
      this.messageIo.outgoingMessageStream.write(message);
      this.transitionTo(this.STATE.SENT_CLIENT_REQUEST);
      message.once('finish', () => {
        request.removeListener('cancel', onCancel);
        request.once('cancel', this._cancelAfterRequestSent);
        this.resetConnectionOnNextRequest = false;
        this.debug.payload(function () {
          return payload.toString('  ');
        });
      });

      const payloadStream = _stream.Readable.from(payload);

      payloadStream.once('error', error => {
        payloadStream.unpipe(message); // Only set a request error if no error was set yet.

        request.error ?? (request.error = error);
        message.ignore = true;
        message.end();
      });
      payloadStream.pipe(message);
    }
  }
  /**
   * Cancel currently executed request.
   */


  cancel() {
    if (!this.request) {
      return false;
    }

    if (this.request.canceled) {
      return false;
    }

    this.request.cancel();
    return true;
  }
  /**
   * Reset the connection to its initial state.
   * Can be useful for connection pool implementations.
   *
   * @param callback
   */


  reset(callback) {
    const request = new _request.default(this.getInitialSql(), err => {
      if (this.config.options.tdsVersion < '7_2') {
        this.inTransaction = false;
      }

      callback(err);
    });
    this.resetConnectionOnNextRequest = true;
    this.execSqlBatch(request);
  }
  /**
   * @private
   */


  currentTransactionDescriptor() {
    return this.transactionDescriptors[this.transactionDescriptors.length - 1];
  }
  /**
   * @private
   */


  getIsolationLevelText(isolationLevel) {
    switch (isolationLevel) {
      case _transaction.ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'read uncommitted';

      case _transaction.ISOLATION_LEVEL.REPEATABLE_READ:
        return 'repeatable read';

      case _transaction.ISOLATION_LEVEL.SERIALIZABLE:
        return 'serializable';

      case _transaction.ISOLATION_LEVEL.SNAPSHOT:
        return 'snapshot';

      default:
        return 'read committed';
    }
  }

}

function isTransientError(error) {
  if (error instanceof _esAggregateError.default) {
    error = error.errors[0];
  }

  return error instanceof _errors.ConnectionError && !!error.isTransient;
}

var _default = Connection;
exports.default = _default;
module.exports = Connection;
Connection.prototype.STATE = {
  INITIALIZED: {
    name: 'Initialized',
    events: {}
  },
  CONNECTING: {
    name: 'Connecting',
    enter: function () {
      this.initialiseConnection();
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_PRELOGIN: {
    name: 'SentPrelogin',
    enter: function () {
      (async () => {
        let messageBuffer = Buffer.alloc(0);
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        for await (const data of message) {
          messageBuffer = Buffer.concat([messageBuffer, data]);
        }

        const preloginPayload = new _preloginPayload.default(messageBuffer);
        this.debug.payload(function () {
          return preloginPayload.toString('  ');
        });

        if (preloginPayload.fedAuthRequired === 1) {
          this.fedAuthRequired = true;
        }

        if (preloginPayload.encryptionString === 'ON' || preloginPayload.encryptionString === 'REQ') {
          if (!this.config.options.encrypt) {
            this.emit('connect', new _errors.ConnectionError("Server requires encryption, set 'encrypt' config option to true.", 'EENCRYPT'));
            return this.close();
          }

          try {
            var _this$routingData;

            this.transitionTo(this.STATE.SENT_TLSSSLNEGOTIATION);
            await this.messageIo.startTls(this.secureContextOptions, ((_this$routingData = this.routingData) === null || _this$routingData === void 0 ? void 0 : _this$routingData.server) ?? this.config.server, this.config.options.trustServerCertificate);
          } catch (err) {
            return this.socketError(err);
          }
        }

        this.sendLogin7Packet();
        const {
          authentication
        } = this.config;

        switch (authentication.type) {
          case 'azure-active-directory-password':
          case 'azure-active-directory-msi-vm':
          case 'azure-active-directory-msi-app-service':
          case 'azure-active-directory-service-principal-secret':
          case 'azure-active-directory-default':
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_FEDAUTH);
            break;

          case 'ntlm':
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            break;

          default:
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            break;
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  REROUTING: {
    name: 'ReRouting',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.REDIRECT);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      reconnect: function () {
        this.transitionTo(this.STATE.CONNECTING);
      }
    }
  },
  TRANSIENT_FAILURE_RETRY: {
    name: 'TRANSIENT_FAILURE_RETRY',
    enter: function () {
      this.curTransientRetryCount++;
      this.cleanupConnection(CLEANUP_TYPE.RETRY);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      retry: function () {
        this.createRetryTimer();
      }
    }
  },
  SENT_TLSSSLNEGOTIATION: {
    name: 'SentTLSSSLNegotiation',
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_STANDARD_LOGIN: {
    name: 'SentLogin7WithStandardLogin',
    enter: function () {
      (async () => {
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end');

        if (handler.loginAckReceived) {
          if (handler.routingData) {
            this.routingData = handler.routingData;
            this.transitionTo(this.STATE.REROUTING);
          } else {
            this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
          }
        } else if (this.loginError) {
          if (isTransientError(this.loginError)) {
            this.debug.log('Initiating retry on transient error');
            this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
          } else {
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
          }
        } else {
          this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
          this.transitionTo(this.STATE.FINAL);
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_NTLM: {
    name: 'SentLogin7WithNTLMLogin',
    enter: function () {
      (async () => {
        while (true) {
          let message;

          try {
            message = await this.messageIo.readMessage();
          } catch (err) {
            return this.socketError(err);
          }

          const handler = new _handler.Login7TokenHandler(this);
          const tokenStreamParser = this.createTokenStreamParser(message, handler);
          await (0, _events.once)(tokenStreamParser, 'end');

          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              return this.transitionTo(this.STATE.REROUTING);
            } else {
              return this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.ntlmpacket) {
            const authentication = this.config.authentication;
            const payload = new _ntlmPayload.default({
              domain: authentication.options.domain,
              userName: authentication.options.userName,
              password: authentication.options.password,
              ntlmpacket: this.ntlmpacket
            });
            this.messageIo.sendMessage(_packet.TYPE.NTLMAUTH_PKT, payload.data);
            this.debug.payload(function () {
              return payload.toString('  ');
            });
            this.ntlmpacket = undefined;
          } else if (this.loginError) {
            if (isTransientError(this.loginError)) {
              this.debug.log('Initiating retry on transient error');
              return this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              return this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            return this.transitionTo(this.STATE.FINAL);
          }
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_FEDAUTH: {
    name: 'SentLogin7Withfedauth',
    enter: function () {
      (async () => {
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end');

        if (handler.loginAckReceived) {
          if (handler.routingData) {
            this.routingData = handler.routingData;
            this.transitionTo(this.STATE.REROUTING);
          } else {
            this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
          }

          return;
        }

        const fedAuthInfoToken = handler.fedAuthInfoToken;

        if (fedAuthInfoToken && fedAuthInfoToken.stsurl && fedAuthInfoToken.spn) {
          const authentication = this.config.authentication;
          const tokenScope = new _url.URL('/.default', fedAuthInfoToken.spn).toString();
          let credentials;

          switch (authentication.type) {
            case 'azure-active-directory-password':
              credentials = new _identity.UsernamePasswordCredential(authentication.options.tenantId ?? 'common', authentication.options.clientId, authentication.options.userName, authentication.options.password);
              break;

            case 'azure-active-directory-msi-vm':
            case 'azure-active-directory-msi-app-service':
              const msiArgs = authentication.options.clientId ? [authentication.options.clientId, {}] : [{}];
              credentials = new _identity.ManagedIdentityCredential(...msiArgs);
              break;

            case 'azure-active-directory-default':
              const args = authentication.options.clientId ? {
                managedIdentityClientId: authentication.options.clientId
              } : {};
              credentials = new _identity.DefaultAzureCredential(args);
              break;

            case 'azure-active-directory-service-principal-secret':
              credentials = new _identity.ClientSecretCredential(authentication.options.tenantId, authentication.options.clientId, authentication.options.clientSecret);
              break;
          }

          let tokenResponse;

          try {
            tokenResponse = await credentials.getToken(tokenScope);
          } catch (err) {
            this.loginError = new _esAggregateError.default([new _errors.ConnectionError('Security token could not be authenticated or authorized.', 'EFEDAUTH'), err]);
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
            return;
          }

          const token = tokenResponse.token;
          this.sendFedAuthTokenMessage(token);
        } else if (this.loginError) {
          if (isTransientError(this.loginError)) {
            this.debug.log('Initiating retry on transient error');
            this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
          } else {
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
          }
        } else {
          this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
          this.transitionTo(this.STATE.FINAL);
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  LOGGED_IN_SENDING_INITIAL_SQL: {
    name: 'LoggedInSendingInitialSql',
    enter: function () {
      (async () => {
        this.sendInitialSql();
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.InitialSqlTokenHandler(this));
        await (0, _events.once)(tokenStreamParser, 'end');
        this.transitionTo(this.STATE.LOGGED_IN);
        this.processedInitialSql();
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function socketError() {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  LOGGED_IN: {
    name: 'LoggedIn',
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_CLIENT_REQUEST: {
    name: 'SentClientRequest',
    enter: function () {
      (async () => {
        var _this$request, _this$request3, _this$request10;

        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        } // request timer is stopped on first data package


        this.clearRequestTimer();
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.RequestTokenHandler(this, this.request)); // If the request was canceled and we have a `cancelTimer`
        // defined, we send a attention message after the
        // request message was fully sent off.
        //
        // We already started consuming the current message
        // (but all the token handlers should be no-ops), and
        // need to ensure the next message is handled by the
        // `SENT_ATTENTION` state.

        if ((_this$request = this.request) !== null && _this$request !== void 0 && _this$request.canceled && this.cancelTimer) {
          return this.transitionTo(this.STATE.SENT_ATTENTION);
        }

        const onResume = () => {
          tokenStreamParser.resume();
        };

        const onPause = () => {
          var _this$request2;

          tokenStreamParser.pause();
          (_this$request2 = this.request) === null || _this$request2 === void 0 ? void 0 : _this$request2.once('resume', onResume);
        };

        (_this$request3 = this.request) === null || _this$request3 === void 0 ? void 0 : _this$request3.on('pause', onPause);

        if (this.request instanceof _request.default && this.request.paused) {
          onPause();
        }

        const onCancel = () => {
          var _this$request4, _this$request5;

          tokenStreamParser.removeListener('end', onEndOfMessage);

          if (this.request instanceof _request.default && this.request.paused) {
            // resume the request if it was paused so we can read the remaining tokens
            this.request.resume();
          }

          (_this$request4 = this.request) === null || _this$request4 === void 0 ? void 0 : _this$request4.removeListener('pause', onPause);
          (_this$request5 = this.request) === null || _this$request5 === void 0 ? void 0 : _this$request5.removeListener('resume', onResume); // The `_cancelAfterRequestSent` callback will have sent a
          // attention message, so now we need to also switch to
          // the `SENT_ATTENTION` state to make sure the attention ack
          // message is processed correctly.

          this.transitionTo(this.STATE.SENT_ATTENTION);
        };

        const onEndOfMessage = () => {
          var _this$request6, _this$request7, _this$request8, _this$request9;

          (_this$request6 = this.request) === null || _this$request6 === void 0 ? void 0 : _this$request6.removeListener('cancel', this._cancelAfterRequestSent);
          (_this$request7 = this.request) === null || _this$request7 === void 0 ? void 0 : _this$request7.removeListener('cancel', onCancel);
          (_this$request8 = this.request) === null || _this$request8 === void 0 ? void 0 : _this$request8.removeListener('pause', onPause);
          (_this$request9 = this.request) === null || _this$request9 === void 0 ? void 0 : _this$request9.removeListener('resume', onResume);
          this.transitionTo(this.STATE.LOGGED_IN);
          const sqlRequest = this.request;
          this.request = undefined;

          if (this.config.options.tdsVersion < '7_2' && sqlRequest.error && this.isSqlBatch) {
            this.inTransaction = false;
          }

          sqlRequest.callback(sqlRequest.error, sqlRequest.rowCount, sqlRequest.rows);
        };

        tokenStreamParser.once('end', onEndOfMessage);
        (_this$request10 = this.request) === null || _this$request10 === void 0 ? void 0 : _this$request10.once('cancel', onCancel);
      })();
    },
    exit: function (nextState) {
      this.clearRequestTimer();
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      }
    }
  },
  SENT_ATTENTION: {
    name: 'SentAttention',
    enter: function () {
      (async () => {
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.AttentionTokenHandler(this, this.request);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end'); // 3.2.5.7 Sent Attention State
        // Discard any data contained in the response, until we receive the attention response

        if (handler.attentionReceived) {
          this.clearCancelTimer();
          const sqlRequest = this.request;
          this.request = undefined;
          this.transitionTo(this.STATE.LOGGED_IN);

          if (sqlRequest.error && sqlRequest.error instanceof _errors.RequestError && sqlRequest.error.code === 'ETIMEOUT') {
            sqlRequest.callback(sqlRequest.error);
          } else {
            sqlRequest.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
          }
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      }
    }
  },
  FINAL: {
    name: 'Final',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.NORMAL);
    },
    events: {
      connectTimeout: function () {// Do nothing, as the timer should be cleaned up.
      },
      message: function () {// Do nothing
      },
      socketError: function () {// Do nothing
      }
    }
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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