/*! @azure/msal-common v7.4.1 2022-09-12 */
'use strict';
import { __awaiter, __generator, __assign } from '../_virtual/_tslib.js';
import { TimeUtils } from '../utils/TimeUtils.js';
import { UrlString } from '../url/UrlString.js';

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */
var KeyLocation;
(function (KeyLocation) {
    KeyLocation["SW"] = "sw";
    KeyLocation["UHW"] = "uhw";
})(KeyLocation || (KeyLocation = {}));
var PopTokenGenerator = /** @class */ (function () {
    function PopTokenGenerator(cryptoUtils) {
        this.cryptoUtils = cryptoUtils;
    }
    /**
     * Generates the req_cnf validated at the RP in the POP protocol for SHR parameters
     * and returns an object containing the keyid, the full req_cnf string and the req_cnf string hash
     * @param request
     * @returns
     */
    PopTokenGenerator.prototype.generateCnf = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var reqCnf, reqCnfString, _a;
            return __generator(this, function (_b) {
                switch (_b.label) {
                    case 0: return [4 /*yield*/, this.generateKid(request)];
                    case 1:
                        reqCnf = _b.sent();
                        reqCnfString = this.cryptoUtils.base64Encode(JSON.stringify(reqCnf));
                        _a = {
                            kid: reqCnf.kid,
                            reqCnfString: reqCnfString
                        };
                        return [4 /*yield*/, this.cryptoUtils.hashString(reqCnfString)];
                    case 2: return [2 /*return*/, (_a.reqCnfHash = _b.sent(),
                            _a)];
                }
            });
        });
    };
    /**
     * Generates key_id for a SHR token request
     * @param request
     * @returns
     */
    PopTokenGenerator.prototype.generateKid = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var kidThumbprint;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.cryptoUtils.getPublicKeyThumbprint(request)];
                    case 1:
                        kidThumbprint = _a.sent();
                        return [2 /*return*/, {
                                kid: kidThumbprint,
                                xms_ksl: KeyLocation.SW
                            }];
                }
            });
        });
    };
    /**
     * Signs the POP access_token with the local generated key-pair
     * @param accessToken
     * @param request
     * @returns
     */
    PopTokenGenerator.prototype.signPopToken = function (accessToken, keyId, request) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                return [2 /*return*/, this.signPayload(accessToken, keyId, request)];
            });
        });
    };
    /**
     * Utility function to generate the signed JWT for an access_token
     * @param payload
     * @param kid
     * @param request
     * @param claims
     * @returns
     */
    PopTokenGenerator.prototype.signPayload = function (payload, keyId, request, claims) {
        return __awaiter(this, void 0, void 0, function () {
            var resourceRequestMethod, resourceRequestUri, shrClaims, shrNonce, resourceUrlString, resourceUrlComponents;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        resourceRequestMethod = request.resourceRequestMethod, resourceRequestUri = request.resourceRequestUri, shrClaims = request.shrClaims, shrNonce = request.shrNonce;
                        resourceUrlString = (resourceRequestUri) ? new UrlString(resourceRequestUri) : undefined;
                        resourceUrlComponents = resourceUrlString === null || resourceUrlString === void 0 ? void 0 : resourceUrlString.getUrlComponents();
                        return [4 /*yield*/, this.cryptoUtils.signJwt(__assign({ at: payload, ts: TimeUtils.nowSeconds(), m: resourceRequestMethod === null || resourceRequestMethod === void 0 ? void 0 : resourceRequestMethod.toUpperCase(), u: resourceUrlComponents === null || resourceUrlComponents === void 0 ? void 0 : resourceUrlComponents.HostNameAndPort, nonce: shrNonce || this.cryptoUtils.createNewGuid(), p: resourceUrlComponents === null || resourceUrlComponents === void 0 ? void 0 : resourceUrlComponents.AbsolutePath, q: (resourceUrlComponents === null || resourceUrlComponents === void 0 ? void 0 : resourceUrlComponents.QueryString) ? [[], resourceUrlComponents.QueryString] : undefined, client_claims: shrClaims || undefined }, claims), keyId, request.correlationId)];
                    case 1: return [2 /*return*/, _a.sent()];
                }
            });
        });
    };
    return PopTokenGenerator;
}());

export { PopTokenGenerator };
//# sourceMappingURL=PopTokenGenerator.js.map
